<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Orchid\Filters\Filterable;
use Orchid\Screen\AsSource;

class Alumno extends Model
{
    use AsSource, Filterable;
    protected $fillable = [
        'user_id',
        'nombre_contacto_emergencia',
        'tel_contacto_emergencia',
        'ci_personas_habilitadas_a_retirarlo',
        'foto',
        'fecha_ingreso',
    ];

    protected $casts = [
        'fecha_ingreso' => 'date',
    ];

    /**
     * Relación con User
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Relación muchos a muchos con Padre
     */
    public function padres(): BelongsToMany
    {
        return $this->belongsToMany(Padre::class, 'padre_alumnos', 'alumno_id', 'padre_id')
            ->withTimestamps();
    }

    /**
     * Relación muchos a muchos con Grupo
     */
    public function grupos(): BelongsToMany
    {
        return $this->belongsToMany(Grupo::class, 'alumno_grupos', 'alumno_id', 'grupo_id')
            ->withTimestamps();
    }

    /**
     * Relación uno a muchos con Calificacion
     */
    public function calificaciones(): HasMany
    {
        return $this->hasMany(Calificacion::class, 'alumno_id');
    }

    /**
     * Relación uno a muchos con CalificacionPeriodo
     */
    public function calificacionesPeriodo(): HasMany
    {
        return $this->hasMany(CalificacionPeriodo::class, 'alumno_id');
    }

    /**
     * Accessor para mostrar el nombre del usuario
     */
    public function getNombreCompletoAttribute(): string
    {
        if (!$this->relationLoaded('user')) {
            $this->load('user');
        }
        return $this->user ? $this->user->name . ' ' . ($this->user->apellido ?? '') : 'ID: ' . $this->id;
    }
}
