<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Orchid\Filters\Filterable;
use Orchid\Screen\AsSource;

class Docente extends Model
{
    use AsSource, Filterable;
    protected $fillable = [
        'user_id',
        'fecha_ingreso',
        'observaciones',
    ];

    protected $casts = [
        'fecha_ingreso' => 'date',
    ];

    /**
     * Relación con User
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Relación muchos a muchos con Asignatura y Grupo
     */
    public function asignaturas(): BelongsToMany
    {
        return $this->belongsToMany(Asignatura::class, 'docente_asignatura_grupos', 'docente_id', 'asignatura_id')
            ->withPivot('grupo_id')
            ->withTimestamps();
    }

    /**
     * Relación muchos a muchos con Grupo a través de Asignatura
     */
    public function grupos(): BelongsToMany
    {
        return $this->belongsToMany(Grupo::class, 'docente_asignatura_grupos', 'docente_id', 'grupo_id')
            ->withPivot('asignatura_id')
            ->withTimestamps();
    }

    /**
     * Relación muchos a muchos con Adscripto
     */
    public function adscriptos(): BelongsToMany
    {
        return $this->belongsToMany(Adscripto::class, 'adscripto_docentes', 'docente_id', 'adscripto_id')
            ->withTimestamps();
    }

    /**
     * Relación uno a muchos con Calificacion
     */
    public function calificaciones(): HasMany
    {
        return $this->hasMany(Calificacion::class, 'docente_id');
    }

    /**
     * Relación uno a muchos con CalificacionPeriodo
     */
    public function calificacionesPeriodo(): HasMany
    {
        return $this->hasMany(CalificacionPeriodo::class, 'docente_id');
    }

    /**
     * Accessor para mostrar el nombre del usuario
     */
    public function getNombreCompletoAttribute(): string
    {
        if (!$this->relationLoaded('user')) {
            $this->load('user');
        }
        return $this->user ? $this->user->name . ' ' . ($this->user->apellido ?? '') : 'ID: ' . $this->id;
    }
}
