<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Orchid\Filters\Types\Like;
use Orchid\Filters\Types\Where;
use Orchid\Filters\Types\WhereDateStartEnd;
use Orchid\Platform\Models\User as Authenticatable;

class User extends Authenticatable
{
    /**
     * Variable estática para almacenar el ID del usuario a incluir en el scope
     * Útil para el formulario de edición de alumnos
     */
    public static $includeUserIdForAlumno = null;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'apellido',
        'email',
        'password',
        'direccion',
        'telefono',
        'celular',
        'fecha_nacimiento',
        'numero_documento',
        'activo',
        'foto',
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
        'permissions',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'permissions'          => 'array',
        'email_verified_at'    => 'datetime',
        'fecha_nacimiento'     => 'date',
        'activo'               => 'boolean',
    ];

    /**
     * The attributes for which you can use filters in url.
     *
     * @var array
     */
    protected $allowedFilters = [
           'id'         => Where::class,
           'name'       => Like::class,
           'email'      => Like::class,
           'updated_at' => WhereDateStartEnd::class,
           'created_at' => WhereDateStartEnd::class,
    ];

    /**
     * The attributes for which can use sort in url.
     *
     * @var array
     */
    protected $allowedSorts = [
        'id',
        'name',
        'email',
        'updated_at',
        'created_at',
    ];

    /**
     * Relación uno a uno con Docente
     */
    public function docente(): HasOne
    {
        return $this->hasOne(Docente::class);
    }

    /**
     * Relación uno a uno con Adscripto
     */
    public function adscripto(): HasOne
    {
        return $this->hasOne(Adscripto::class);
    }

    /**
     * Relación uno a uno con Padre
     */
    public function padre(): HasOne
    {
        return $this->hasOne(Padre::class);
    }

    /**
     * Relación uno a uno con Alumno
     */
    public function alumno(): HasOne
    {
        return $this->hasOne(Alumno::class);
    }

    /**
     * Relación muchos a muchos con Rol
     */
    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Rol::class, 'role_users', 'user_id', 'role_id');
    }

    /**
     * Scope para filtrar usuarios con rol "Padre"
     */
    public function scopeWithPadreRole($query)
    {
        return $query->whereHas('roles', function ($q) {
            $q->where('slug', 'padre');
        });
    }

    /**
     * Scope para filtrar usuarios con rol "Alumno"
     */
    public function scopeWithAlumnoRole($query)
    {
        return $query->whereHas('roles', function ($q) {
            $q->where('slug', 'alumno');
        });
    }

    /**
     * Scope para usuarios con rol Alumno que no tienen registro de alumno
     * Combina el filtro de rol y la exclusión de registros existentes
     */
    public function scopeAvailableForAlumno($query)
    {
        $query->whereHas('roles', function ($q) {
            $q->where('slug', 'alumno');
        });
        
        if (static::$includeUserIdForAlumno) {
            return $query->where(function ($q) {
                $q->whereDoesntHave('alumno')
                  ->orWhere('id', static::$includeUserIdForAlumno);
            });
        }
        return $query->whereDoesntHave('alumno');
    }

    /**
     * Scope para filtrar usuarios con rol "Docente"
     */
    public function scopeWithDocenteRole($query)
    {
        return $query->whereHas('roles', function ($q) {
            $q->where('slug', 'docente');
        });
    }

    /**
     * Scope para filtrar usuarios con rol "Adscripto"
     */
    public function scopeWithAdscriptoRole($query)
    {
        return $query->whereHas('roles', function ($q) {
            $q->where('slug', 'adscripto');
        });
    }

    /**
     * Scope para excluir usuarios que ya tienen un registro de alumno
     * Útil para evitar duplicados al crear nuevos alumnos
     * Si se establece User::$includeUserIdForAlumno, incluye ese usuario (útil al editar)
     */
    public function scopeWithoutAlumnoRecord($query)
    {
        if (static::$includeUserIdForAlumno) {
            return $query->where(function ($q) {
                $q->whereDoesntHave('alumno')
                  ->orWhere('id', static::$includeUserIdForAlumno);
            });
        }
        return $query->whereDoesntHave('alumno');
    }

    /**
     * Accessor para mostrar el nombre completo del usuario
     */
    public function getNombreCompletoAttribute(): string
    {
        return trim($this->name . ' ' . ($this->apellido ?? ''));
    }
}
