<?php

declare(strict_types=1);

namespace App\Orchid\Layouts\Alumno;

use App\Models\Grupo;
use App\Models\User;
use Orchid\Screen\Field;
use Orchid\Screen\Fields\DateTimer;
use Orchid\Screen\Fields\Input;
use Orchid\Screen\Fields\Relation;
use Orchid\Screen\Fields\TextArea;
use Orchid\Screen\Layouts\Rows;

class AlumnoEditLayout extends Rows
{
    public function fields(): array
    {
        // Obtener el alumno desde el query
        $alumno = $this->query->get('alumno');
        $includeUserId = ($alumno && $alumno->exists && $alumno->user_id) ? $alumno->user_id : null;
        
        // Establecer la variable estática para que el scope pueda acceder a ella
        User::$includeUserIdForAlumno = $includeUserId;
        
        return [
            Relation::make('alumno.user_id')
                ->fromModel(User::class, 'name')
                ->applyScope('availableForAlumno')
                ->title(__('Usuario'))
                ->required()
                ->help(__('Seleccione el usuario que será alumno. Solo se muestran usuarios con rol "Alumno" que aún no tienen registro de alumno.')),

            Relation::make('alumno.grupos.')
                ->fromModel(Grupo::class, 'nombre_grupo')
                ->multiple()
                ->title(__('Grupos'))
                ->help(__('Seleccione uno o más grupos para el alumno'))
                ->chunk(20),

            Relation::make('alumno.padres.')
                ->fromModel(User::class, 'name')
                ->applyScope('withPadreRole')
                ->multiple()
                ->title(__('Padres/Tutores'))
                ->help(__('Seleccione uno o más usuarios con rol "Padre" que serán padres/tutores del alumno. Si el usuario no tiene registro de padre, se creará automáticamente. Nota: El usuario del alumno no puede ser su propio padre.'))
                ->searchColumns('name', 'email', 'apellido')
                ->displayAppend('nombre_completo')
                ->chunk(20),

            Input::make('alumno.nombre_contacto_emergencia')
                ->type('text')
                ->max(255)
                ->title(__('Nombre Contacto Emergencia'))
                ->placeholder(__('Nombre del contacto de emergencia')),

            Input::make('alumno.tel_contacto_emergencia')
                ->type('text')
                ->max(50)
                ->title(__('Teléfono Contacto Emergencia'))
                ->placeholder(__('Teléfono del contacto de emergencia')),

            TextArea::make('alumno.ci_personas_habilitadas_a_retirarlo')
                ->title(__('CI Personas Habilitadas a Retirarlo'))
                ->rows(3)
                ->placeholder(__('Cédulas de identidad de personas autorizadas')),

            DateTimer::make('alumno.fecha_ingreso')
                ->title(__('Fecha de Ingreso'))
                ->format('Y-m-d')
                ->allowInput(),
        ];
    }
}

