<?php

declare(strict_types=1);

namespace App\Orchid;

use Illuminate\Support\Facades\Auth;
use Orchid\Platform\Dashboard;
use Orchid\Platform\ItemPermission;
use Orchid\Platform\OrchidServiceProvider;
use Orchid\Screen\Actions\Menu;
use Orchid\Support\Color;

class PlatformProvider extends OrchidServiceProvider
{
    /**
     * Bootstrap the application services.
     *
     * @param Dashboard $dashboard
     *
     * @return void
     */
    public function boot(Dashboard $dashboard): void
    {
        parent::boot($dashboard);

        // ...
    }

    /**
     * Register the application menu.
     *
     * @return Menu[]
     */
    public function menu(): array
    {
        $user = Auth::user();
        $isDocente = $user && $user->docente;
        
        $menu = [
            // 1. Control de accesos
            Menu::make(__('Users'))
                ->icon('bs.people')
                ->route('platform.systems.users')
                ->permission('platform.systems.users')
                ->title(__('Control de accesos')),

            Menu::make(__('Roles'))
                ->icon('bs.shield')
                ->route('platform.systems.roles')
                ->permission('platform.systems.roles')
                ->divider(),

            // 2. Personal
            Menu::make(__('Docentes'))
                ->icon('bs.person-badge')
                ->route('platform.systems.docentes')
                ->permission('platform.systems.docentes')
                ->title(__('Personal')),

            Menu::make(__('Alumnos'))
                ->icon('bs.person')
                ->route('platform.systems.alumnos')
                ->permission('platform.systems.alumnos'),

            Menu::make(__('Adscriptos'))
                ->icon('bs.person-check')
                ->route('platform.systems.adscriptos')
                ->permission('platform.systems.adscriptos'),

            Menu::make(__('Padres'))
                ->icon('bs.person-heart')
                ->route('platform.systems.padres')
                ->permission('platform.systems.padres')
                ->divider(),

            // 3. Gestión Académica
            Menu::make(__('Colegios'))
                ->icon('bs.building')
                ->route('platform.systems.colegios')
                ->permission('platform.systems.colegios')
                ->title(__('Gestión Académica')),

            Menu::make(__('Ciclos'))
                ->icon('bs.calendar')
                ->route('platform.systems.ciclos')
                ->permission('platform.systems.ciclos'),

            Menu::make(__('Turnos'))
                ->icon('bs.clock')
                ->route('platform.systems.turnos')
                ->permission('platform.systems.turnos'),

            Menu::make(__('Asignaturas'))
                ->icon('bs.book')
                ->route('platform.systems.asignaturas')
                ->permission('platform.systems.asignaturas'),

            Menu::make(__('Grupos'))
                ->icon('bs.people-fill')
                ->route('platform.systems.grupos')
                ->permission('platform.systems.grupos'),

            Menu::make(__('Períodos Académicos'))
                ->icon('bs.calendar-range')
                ->route('platform.systems.periodos')
                ->permission('platform.systems.periodos')
                ->divider(),

            // 4. Evaluaciones
            Menu::make(__('Calificaciones'))
                ->icon('bs.star')
                ->route('platform.systems.calificaciones')
                ->permission('platform.systems.calificaciones')
                ->title(__('Evaluaciones')),

            Menu::make(__('Cerrar promedios'))
                ->icon('bs.calendar-check')
                ->route('platform.systems.calificaciones-periodo')
                ->permission('platform.systems.calificaciones-periodo')
                ->divider(),

        ];
        
        // Si NO es docente, agregar las secciones Navigation y Docs
        if (!$isDocente) {
            $menu = array_merge($menu, [
                // 5. Navigation
                Menu::make('Get Started')
                    ->icon('bs.book')
                    ->title('Navigation')
                    ->route(config('platform.index')),

                Menu::make('Sample Screen')
                    ->icon('bs.collection')
                    ->route('platform.example')
                    ->badge(fn () => 6),

                Menu::make('Form Elements')
                    ->icon('bs.card-list')
                    ->route('platform.example.fields')
                    ->active('*/examples/form/*'),

                Menu::make('Layouts Overview')
                    ->icon('bs.window-sidebar')
                    ->route('platform.example.layouts'),

                Menu::make('Grid System')
                    ->icon('bs.columns-gap')
                    ->route('platform.example.grid'),

                Menu::make('Charts')
                    ->icon('bs.bar-chart')
                    ->route('platform.example.charts'),

                Menu::make('Cards')
                    ->icon('bs.card-text')
                    ->route('platform.example.cards')
                    ->divider(),

                // 6. Docs
                Menu::make('Documentation')
                    ->title('Docs')
                    ->icon('bs.box-arrow-up-right')
                    ->url('https://orchid.software/en/docs')
                    ->target('_blank'),

                Menu::make('Changelog')
                    ->icon('bs.box-arrow-up-right')
                    ->url('https://github.com/orchidsoftware/platform/blob/master/CHANGELOG.md')
                    ->target('_blank')
                    ->badge(fn () => Dashboard::version(), Color::DARK),
            ]);
        }
        
        return $menu;
    }

    /**
     * Register permissions for the application.
     *
     * @return ItemPermission[]
     */
    public function permissions(): array
    {
        return [
            ItemPermission::group(__('System'))
                ->addPermission('platform.systems.roles', __('Roles'))
                ->addPermission('platform.systems.users', __('Users')),
            
            ItemPermission::group(__('Gestión Académica'))
                ->addPermission('platform.systems.colegios', __('Colegios'))
                ->addPermission('platform.systems.ciclos', __('Ciclos'))
                ->addPermission('platform.systems.turnos', __('Turnos'))
                ->addPermission('platform.systems.asignaturas', __('Asignaturas'))
                ->addPermission('platform.systems.grupos', __('Grupos'))
                ->addPermission('platform.systems.periodos', __('Períodos Académicos')),
            
            ItemPermission::group(__('Personal'))
                ->addPermission('platform.systems.docentes', __('Docentes'))
                ->addPermission('platform.systems.alumnos', __('Alumnos'))
                ->addPermission('platform.systems.adscriptos', __('Adscriptos'))
                ->addPermission('platform.systems.padres', __('Padres')),
            
            ItemPermission::group(__('Evaluaciones'))
                ->addPermission('platform.systems.calificaciones', __('Calificaciones'))
                ->addPermission('platform.systems.calificaciones-periodo', __('Cerrar promedios')),
        ];
    }
}
