<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Alumno;

use App\Models\Alumno;
use App\Models\Padre;
use App\Models\User;
use App\Orchid\Layouts\Alumno\AlumnoEditLayout;
use Illuminate\Http\Request;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\Screen;
use Orchid\Support\Color;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Toast;

class AlumnoEditScreen extends Screen
{
    public $alumno;

    public function query(Alumno $alumno): iterable
    {
        $alumno->load(['user', 'grupos', 'padres.user']);
        
        // Convertir los padres a user_ids para el campo Relation
        // Si el alumno es nuevo o no tiene padres, retornar array vacío
        $padresUserIds = $alumno->exists && $alumno->padres->isNotEmpty() 
            ? $alumno->padres->pluck('user_id')->toArray() 
            : [];
        
        return [
            'alumno' => $alumno,
            'alumno.padres' => $padresUserIds,
            'alumno_user_id' => $alumno->user_id, // Para excluir en el campo de padres
            'alumno_id' => $alumno->id, // Para filtrar usuarios disponibles en el campo Usuario
        ];
    }

    public function name(): ?string
    {
        return $this->alumno->exists ? 'Editar Alumno' : 'Crear Alumno';
    }

    public function description(): ?string
    {
        return 'Información del alumno.';
    }

    public function permission(): ?iterable
    {
        return null; // Sin restricción de permisos
    }

    public function commandBar(): iterable
    {
        return [
            Button::make(__('Eliminar'))
                ->icon('bs.trash3')
                ->confirm(__('¿Está seguro de eliminar este alumno?'))
                ->method('remove')
                ->canSee($this->alumno->exists),
            Button::make(__('Guardar'))
                ->icon('bs.check-circle')
                ->method('save'),
        ];
    }

    public function layout(): iterable
    {
        return [
            Layout::block(AlumnoEditLayout::class)
                ->title(__('Información del Alumno'))
                ->description(__('Ingrese la información del alumno.'))
                ->commands(
                    Button::make(__('Guardar'))
                        ->type(Color::BASIC)
                        ->icon('bs.check-circle')
                        ->canSee($this->alumno->exists)
                        ->method('save')
                ),
        ];
    }

    public function save(Alumno $alumno, Request $request)
    {
        $request->validate([
            'alumno.user_id' => 'required|exists:users,id|unique:alumnos,user_id,' . $alumno->id,
            'alumno.nombre_contacto_emergencia' => 'nullable|string|max:255',
            'alumno.tel_contacto_emergencia' => 'nullable|string|max:50',
            'alumno.ci_personas_habilitadas_a_retirarlo' => 'nullable|string',
            'alumno.foto' => 'nullable|string|max:255',
            'alumno.fecha_ingreso' => 'nullable|date',
            'alumno.grupos' => 'nullable|array',
            'alumno.grupos.*' => 'exists:grupos,id',
            'alumno.padres' => 'nullable|array',
            'alumno.padres.*' => 'exists:users,id',
        ]);

        $alumno->fill($request->get('alumno'))->save();

        // Sincronizar grupos
        $grupos = $request->input('alumno.grupos', []);
        $grupos = is_array($grupos) ? array_filter($grupos) : [];
        $alumno->grupos()->sync($grupos);

        // Sincronizar padres - convertir user_ids a padre_ids
        $padresUserIds = $request->input('alumno.padres', []);
        $padresUserIds = is_array($padresUserIds) ? array_filter($padresUserIds) : [];
        
        // Excluir al usuario del alumno de la lista de padres (no puede ser padre de sí mismo)
        $alumnoUserId = $alumno->user_id;
        $padresUserIds = array_filter($padresUserIds, fn($userId) => $userId != $alumnoUserId);
        
        // Obtener o crear los registros Padre para los user_ids seleccionados
        $padreIds = [];
        foreach ($padresUserIds as $userId) {
            // Validar que el usuario existe y tiene el rol "Padre"
            $user = User::find($userId);
            if (!$user) {
                continue;
            }
            
            // Verificar que el usuario tenga el rol "Padre"
            $hasPadreRole = $user->roles()->where('slug', 'padre')->exists();
            if (!$hasPadreRole) {
                Toast::warning(__('El usuario seleccionado no tiene el rol "Padre". Solo se pueden asignar usuarios con rol "Padre" como padres/tutores.'));
                continue;
            }
            
            // Buscar si ya existe un registro Padre para este usuario
            $padre = Padre::firstOrCreate(
                ['user_id' => $userId],
                ['tipo_relacion' => 'Padre'] // Valor por defecto
            );
            $padreIds[] = $padre->id;
        }
        
        // Sincronizar las relaciones
        $alumno->padres()->sync($padreIds);

        Toast::info(__('Alumno guardado correctamente'));
        return redirect()->route('platform.systems.alumnos');
    }

    public function remove(Alumno $alumno)
    {
        $alumno->delete();
        Toast::info(__('Alumno eliminado correctamente'));
        return redirect()->route('platform.systems.alumnos');
    }
}

