<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Alumno;

use App\Models\Alumno;
use App\Orchid\Presenters\UserPresenter;
use Illuminate\Support\Facades\Storage;
use Orchid\Screen\Actions\Link;
use Orchid\Screen\Screen;
use Orchid\Screen\Sight;
use Orchid\Support\Facades\Layout;

class AlumnoShowScreen extends Screen
{
    public $alumno;

    public function query(Alumno $alumno): iterable
    {
        $this->alumno = $alumno;
        $alumno->load(['user', 'grupos', 'padres.user']);
        
        return [
            'alumno' => $alumno,
        ];
    }

    public function name(): ?string
    {
        return 'Detalles del Alumno';
    }

    public function description(): ?string
    {
        return $this->alumno->user 
            ? 'Información completa de ' . $this->alumno->user->name . ' ' . ($this->alumno->user->apellido ?? '')
            : 'Información completa del alumno.';
    }

    public function permission(): ?iterable
    {
        return null; // Sin restricción de permisos
    }

    public function commandBar(): iterable
    {
        return [
            Link::make(__('Volver'))
                ->icon('bs.arrow-left')
                ->route('platform.systems.alumnos'),
            Link::make(__('Editar'))
                ->icon('bs.pencil')
                ->route('platform.systems.alumnos.edit', $this->alumno->id),
        ];
    }

    public function layout(): iterable
    {
        // Obtener la URL de la foto del usuario
        $fotoUrl = null;
        if ($this->alumno->user) {
            $userPresenter = new UserPresenter($this->alumno->user);
            $fotoUrl = $userPresenter->image();
        }
        
        $nombreCompleto = $this->alumno->user 
            ? $this->alumno->user->name . ' ' . ($this->alumno->user->apellido ?? '') 
            : 'Alumno';
        
        return [
            // Bloque con la foto de perfil
            Layout::block(Layout::view('alumno-foto', [
                'fotoUrl' => $fotoUrl,
                'nombreCompleto' => $nombreCompleto,
            ]))
                ->title(__('Foto de Perfil'))
                ->description(__('Foto de perfil del alumno')),
            
            Layout::legend('alumno', [
                Sight::make('id', __('ID')),
                
                Sight::make('user_id', __('Usuario'))
                    ->render(fn (Alumno $alumno) => $alumno->user 
                        ? $alumno->user->name . ' ' . ($alumno->user->apellido ?? '') 
                        : '-'),
                
                Sight::make('email', __('Email'))
                    ->render(fn (Alumno $alumno) => $alumno->user && $alumno->user->email 
                        ? $alumno->user->email 
                        : '-'),
                
                Sight::make('grupos', __('Grupos'))
                    ->render(function (Alumno $alumno) {
                        if ($alumno->grupos->isEmpty()) {
                            return '-';
                        }
                        return $alumno->grupos->pluck('nombre_grupo')->implode(', ');
                    }),
                
                Sight::make('padres', __('Padres/Tutores'))
                    ->render(function (Alumno $alumno) {
                        if ($alumno->padres->isEmpty()) {
                            return '-';
                        }
                        return $alumno->padres->map(function ($padre) {
                            return $padre->user 
                                ? $padre->user->name . ' ' . ($padre->user->apellido ?? '') 
                                : '-';
                        })->filter()->implode(', ');
                    }),
                
                Sight::make('nombre_contacto_emergencia', __('Nombre Contacto Emergencia'))
                    ->render(fn (Alumno $alumno) => $alumno->nombre_contacto_emergencia ?: '-'),
                
                Sight::make('tel_contacto_emergencia', __('Teléfono Contacto Emergencia'))
                    ->render(fn (Alumno $alumno) => $alumno->tel_contacto_emergencia ?: '-'),
                
                Sight::make('ci_personas_habilitadas_a_retirarlo', __('CI Personas Habilitadas a Retirarlo'))
                    ->render(fn (Alumno $alumno) => $alumno->ci_personas_habilitadas_a_retirarlo ?: '-'),
                
                Sight::make('fecha_ingreso', __('Fecha de Ingreso'))
                    ->render(fn (Alumno $alumno) => $alumno->fecha_ingreso 
                        ? $alumno->fecha_ingreso->format('d/m/Y') 
                        : '-'),
                
                Sight::make('created_at', __('Fecha de Creación'))
                    ->render(fn (Alumno $alumno) => $alumno->created_at 
                        ? $alumno->created_at->format('d/m/Y H:i') 
                        : '-'),
                
                Sight::make('updated_at', __('Última Actualización'))
                    ->render(fn (Alumno $alumno) => $alumno->updated_at 
                        ? $alumno->updated_at->format('d/m/Y H:i') 
                        : '-'),
            ])->title(__('Información del Alumno')),
        ];
    }
}

