<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Asignatura;

use App\Models\Asignatura;
use App\Orchid\Layouts\Asignatura\AsignaturaEditLayout;
use Illuminate\Http\Request;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\Screen;
use Orchid\Support\Color;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Toast;

class AsignaturaEditScreen extends Screen
{
    public $asignatura;

    public function query(Asignatura $asignatura): iterable
    {
        return ['asignatura' => $asignatura];
    }

    public function name(): ?string
    {
        return $this->asignatura->exists ? 'Editar Asignatura' : 'Crear Asignatura';
    }

    public function description(): ?string
    {
        return 'Información de la asignatura.';
    }

    public function permission(): ?iterable
    {
        return null; // Sin restricción de permisos
    }

    public function commandBar(): iterable
    {
        return [
            Button::make(__('Eliminar'))
                ->icon('bs.trash3')
                ->confirm(__('¿Está seguro de eliminar esta asignatura?'))
                ->method('remove')
                ->canSee($this->asignatura->exists),
            Button::make(__('Guardar'))
                ->icon('bs.check-circle')
                ->method('save'),
        ];
    }

    public function layout(): iterable
    {
        return [
            Layout::block(AsignaturaEditLayout::class)
                ->title(__('Información de la Asignatura'))
                ->description(__('Ingrese la información de la asignatura.'))
                ->commands(
                    Button::make(__('Guardar'))
                        ->type(Color::BASIC)
                        ->icon('bs.check-circle')
                        ->canSee($this->asignatura->exists)
                        ->method('save')
                ),
        ];
    }

    public function save(Asignatura $asignatura, Request $request)
    {
        $request->validate([
            'asignatura.nombre' => 'required|string|max:255',
        ]);

        $asignatura->fill($request->get('asignatura'))->save();
        Toast::info(__('Asignatura guardada correctamente'));
        return redirect()->route('platform.systems.asignaturas');
    }

    public function remove(Asignatura $asignatura)
    {
        $asignatura->delete();
        Toast::info(__('Asignatura eliminada correctamente'));
        return redirect()->route('platform.systems.asignaturas');
    }
}

