<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Calificacion;

use App\Models\Alumno;
use App\Models\Calificacion;
use App\Models\Grupo;
use App\Orchid\Layouts\Calificacion\CalificacionAlumnoLayout;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Orchid\Screen\Actions\Link;
use Orchid\Screen\Screen;
use Orchid\Support\Facades\Toast;

class CalificacionAlumnoScreen extends Screen
{
    public $alumno;
    public $grupo;

    public function query(Alumno $alumno, Request $request): iterable
    {
        $this->alumno = $alumno;
        $this->alumno->load('user');
        
        $grupoId = $request->get('grupo_id');
        if ($grupoId) {
            $this->grupo = Grupo::find($grupoId);
        }
        
        // Construir query de calificaciones
        $calificacionesQuery = Calificacion::with(['alumno.user', 'asignatura', 'docente.user'])
            ->where('alumno_id', $alumno->id);
        
        // Si el usuario autenticado es un docente, filtrar solo sus calificaciones
        $user = Auth::user();
        if ($user && $user->docente) {
            $calificacionesQuery->where('docente_id', $user->docente->id);
        }
        
        return [
            'alumno' => $this->alumno,
            'grupo' => $this->grupo,
            'grupo_id' => $grupoId,
            'calificaciones' => $calificacionesQuery
                ->orderBy('anio', 'desc')
                ->orderBy('mes', 'desc')
                ->paginate(),
        ];
    }

    public function name(): ?string
    {
        $nombreAlumno = $this->alumno && $this->alumno->user 
            ? $this->alumno->user->name . ' ' . ($this->alumno->user->apellido ?? '') 
            : 'Alumno';
        return 'Calificaciones de ' . $nombreAlumno;
    }

    public function description(): ?string
    {
        return 'Lista de calificaciones del alumno.';
    }

    public function permission(): ?iterable
    {
        return null;
    }

    public function commandBar(): iterable
    {
        $actions = [
            Link::make(__('Agregar Calificación'))
                ->icon('bs.plus-circle')
                ->route('platform.systems.calificaciones.create', [
                    'alumno_id' => $this->alumno->id,
                ]),
        ];

        if ($this->grupo) {
            $actions[] = Link::make(__('Volver a Alumnos del Grupo'))
                ->icon('bs.arrow-left')
                ->route('platform.systems.grupos.alumnos', $this->grupo->id);
        } else {
            $actions[] = Link::make(__('Volver a Calificaciones'))
                ->icon('bs.arrow-left')
                ->route('platform.systems.calificaciones');
        }

        return $actions;
    }

    public function layout(): iterable
    {
        return [CalificacionAlumnoLayout::class];
    }

    public function remove(Request $request)
    {
        $user = Auth::user();
        
        // Si el usuario es docente, verificar que la calificación sea suya
        if ($user && $user->docente) {
            $docenteIdRequest = $request->get('docente_id');
            if ($docenteIdRequest != $user->docente->id) {
                Toast::error(__('No tiene permiso para eliminar esta calificación'));
                $alumnoId = $request->get('alumno_id');
                $grupoId = $request->get('grupo_id');
                
                if ($grupoId) {
                    return redirect()->route('platform.systems.alumnos.calificaciones', [
                        'alumno' => $alumnoId,
                        'grupo_id' => $grupoId,
                    ]);
                } else {
                    return redirect()->route('platform.systems.alumnos.calificaciones', ['alumno' => $alumnoId]);
                }
            }
        }
        
        $query = Calificacion::where('alumno_id', $request->get('alumno_id'))
            ->where('asignatura_id', $request->get('asignatura_id'))
            ->where('docente_id', $request->get('docente_id'))
            ->where('mes', $request->get('mes'))
            ->where('anio', $request->get('anio'));
        
        // Si viene numero, eliminar solo esa calificación específica
        if ($request->has('numero')) {
            $query->where('numero', $request->get('numero'));
        }
        
        $query->delete();
        Toast::info(__('Calificación eliminada correctamente'));
        
        $alumnoId = $request->get('alumno_id');
        $grupoId = $request->get('grupo_id');
        
        if ($grupoId) {
            return redirect()->route('platform.systems.alumnos.calificaciones', [
                'alumno' => $alumnoId,
                'grupo_id' => $grupoId,
            ]);
        } else {
            return redirect()->route('platform.systems.alumnos.calificaciones', ['alumno' => $alumnoId]);
        }
    }
}

