<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Calificacion;

use App\Models\Calificacion;
use App\Orchid\Layouts\Calificacion\CalificacionEditLayout;
use Illuminate\Http\Request;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\Screen;
use Orchid\Support\Color;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Toast;

class CalificacionEditScreen extends Screen
{
    public $calificacion;

    public function query(Request $request): iterable
    {
        $query = Calificacion::where('alumno_id', $request->get('alumno_id'))
            ->where('asignatura_id', $request->get('asignatura_id'))
            ->where('docente_id', $request->get('docente_id'))
            ->where('mes', $request->get('mes'))
            ->where('anio', $request->get('anio'));
        
        // Si viene numero, buscar esa calificación específica
        if ($request->has('numero')) {
            $query->where('numero', $request->get('numero'));
        } else {
            // Si no viene numero, tomar la primera (para compatibilidad)
            $query->orderBy('numero', 'asc');
        }
        
        $this->calificacion = $query->first() ?? new Calificacion();
        
        // Si es nuevo y no tiene numero, asignar 1 por defecto
        if (!$this->calificacion->exists && !$this->calificacion->numero) {
            $this->calificacion->numero = 1;
        }
        
        $this->calificacion->load(['alumno.user', 'asignatura', 'docente.user']);
        return ['calificacion' => $this->calificacion];
    }

    public function name(): ?string
    {
        return $this->calificacion->exists ? 'Editar Calificación' : 'Crear Calificación';
    }

    public function description(): ?string
    {
        return 'Información de la calificación.';
    }

    public function permission(): ?iterable
    {
        return null;
    }

    public function commandBar(): iterable
    {
        return [
            Button::make(__('Eliminar'))
                ->icon('bs.trash3')
                ->confirm(__('¿Está seguro?'))
                ->method('remove')
                ->canSee($this->calificacion->exists),
            Button::make(__('Guardar'))
                ->icon('bs.check-circle')
                ->method('save'),
        ];
    }

    public function layout(): iterable
    {
        return [
            Layout::block(CalificacionEditLayout::class)
                ->title(__('Información de la Calificación'))
                ->commands(
                    Button::make(__('Guardar'))
                        ->type(Color::BASIC)
                        ->icon('bs.check-circle')
                        ->canSee($this->calificacion->exists)
                        ->method('save')
                ),
        ];
    }

    public function save(Request $request)
    {
        $request->validate([
            'calificacion.alumno_id' => 'required|exists:alumnos,id',
            'calificacion.asignatura_id' => 'required|exists:asignaturas,id',
            'calificacion.docente_id' => 'required|exists:docentes,id',
            'calificacion.mes' => 'required|integer|min:1|max:12',
            'calificacion.anio' => 'required|integer|min:2000|max:2100',
            'calificacion.nota' => 'nullable|numeric|min:0|max:10',
            'calificacion.comentario' => 'nullable|string',
        ]);

        $calificacionData = $request->get('calificacion');
        
        // Construir los criterios de búsqueda (clave primaria compuesta)
        $searchCriteria = [
            'alumno_id' => $calificacionData['alumno_id'],
            'asignatura_id' => $calificacionData['asignatura_id'],
            'docente_id' => $calificacionData['docente_id'],
            'mes' => $calificacionData['mes'],
            'anio' => $calificacionData['anio'],
        ];
        
        // Si viene numero en el request (puede venir como parámetro de URL o en calificacion), significa que estamos editando una calificación específica
        $numero = $request->get('numero') ?? (isset($calificacionData['numero']) ? (int)$calificacionData['numero'] : null);
        if ($numero !== null) {
            $numero = (int)$numero;
        }
        
        if ($numero !== null) {
            // Estamos editando una calificación específica (viene desde el formulario de edición)
            $calificacion = Calificacion::where('alumno_id', $searchCriteria['alumno_id'])
                ->where('asignatura_id', $searchCriteria['asignatura_id'])
                ->where('docente_id', $searchCriteria['docente_id'])
                ->where('mes', $searchCriteria['mes'])
                ->where('anio', $searchCriteria['anio'])
                ->where('numero', $numero)
                ->first();
            
            if ($calificacion) {
                // Actualizar la calificación específica
                Calificacion::where('alumno_id', $searchCriteria['alumno_id'])
                    ->where('asignatura_id', $searchCriteria['asignatura_id'])
                    ->where('docente_id', $searchCriteria['docente_id'])
                    ->where('mes', $searchCriteria['mes'])
                    ->where('anio', $searchCriteria['anio'])
                    ->where('numero', $numero)
                    ->update([
                        'nota' => $calificacionData['nota'] ?? null,
                        'comentario' => $calificacionData['comentario'] ?? null,
                    ]);
            } else {
                // Si no existe con ese numero, crear uno nuevo
                Calificacion::create([
                    'alumno_id' => $searchCriteria['alumno_id'],
                    'asignatura_id' => $searchCriteria['asignatura_id'],
                    'docente_id' => $searchCriteria['docente_id'],
                    'mes' => $searchCriteria['mes'],
                    'anio' => $searchCriteria['anio'],
                    'numero' => $numero,
                    'nota' => $calificacionData['nota'] ?? null,
                    'comentario' => $calificacionData['comentario'] ?? null,
                ]);
            }
        } else {
            // Estamos creando una nueva calificación - calcular el siguiente numero disponible
            $maxNumero = Calificacion::where('alumno_id', $searchCriteria['alumno_id'])
                ->where('asignatura_id', $searchCriteria['asignatura_id'])
                ->where('docente_id', $searchCriteria['docente_id'])
                ->where('mes', $searchCriteria['mes'])
                ->where('anio', $searchCriteria['anio'])
                ->max('numero') ?? 0;
            
            $nuevoNumero = $maxNumero + 1;
            
            // Crear un nuevo registro con el siguiente numero
            Calificacion::create([
                'alumno_id' => $searchCriteria['alumno_id'],
                'asignatura_id' => $searchCriteria['asignatura_id'],
                'docente_id' => $searchCriteria['docente_id'],
                'mes' => $searchCriteria['mes'],
                'anio' => $searchCriteria['anio'],
                'numero' => $nuevoNumero,
                'nota' => $calificacionData['nota'] ?? null,
                'comentario' => $calificacionData['comentario'] ?? null,
            ]);
        }

        Toast::info(__('Calificación guardada correctamente'));
        return redirect()->route('platform.systems.calificaciones');
    }

    public function remove(Request $request)
    {
        $query = Calificacion::where('alumno_id', $request->get('alumno_id'))
            ->where('asignatura_id', $request->get('asignatura_id'))
            ->where('docente_id', $request->get('docente_id'))
            ->where('mes', $request->get('mes'))
            ->where('anio', $request->get('anio'));
        
        // Si viene numero, eliminar solo esa calificación específica
        if ($request->has('numero')) {
            $query->where('numero', $request->get('numero'));
        }
        
        $query->delete();
        Toast::info(__('Calificación eliminada correctamente'));
        return redirect()->route('platform.systems.calificaciones');
    }
}
