<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Calificacion;

use App\Models\Alumno;
use App\Models\Asignatura;
use App\Models\Calificacion;
use App\Models\CalificacionPeriodo;
use App\Models\Periodo;
use App\Models\Grupo;
use App\Orchid\Layouts\Calificacion\VerCalificacionesMensualesLayout;
use App\Orchid\Layouts\Calificacion\PromediosPeriodoLayout;
use App\Orchid\Layouts\Calificacion\SelectorAsignaturaAnioLayout;
use Illuminate\Http\Request;
use Orchid\Screen\Actions\Link;
use Orchid\Screen\Screen;
use Orchid\Support\Facades\Layout;

class VerCalificacionesAlumnoScreen extends Screen
{
	public $alumno;
	public $grupo;

	public function query(Alumno $alumno, Request $request): iterable
	{
		$this->alumno = $alumno->load('user');

		$grupoId = (int) $request->get('grupo_id');
		$this->grupo = $grupoId ? Grupo::find($grupoId) : null;

		$asignaturaId = $request->integer('asignatura_id') ?: null;

		// Asignaturas disponibles para el grupo (si hay grupo) o todas en su defecto
		$asignaturasQuery = Asignatura::query();
		if ($this->grupo) {
			$asignaturasQuery->whereHas('grupos', function ($q) use ($grupoId) {
				$q->where('grupos.id', $grupoId);
			});
		}
		$asignaturas = $asignaturasQuery->orderBy('nombre')->get(['id', 'nombre']);

		// Fallback: si no hay asignaturas vinculadas al grupo, usar asignaturas donde el alumno tenga calificaciones
		if ($asignaturas->isEmpty()) {
			$idsAsignaturasAlumno = Calificacion::query()
				->where('alumno_id', $alumno->id)
				->distinct()
				->pluck('asignatura_id')
				->filter()
				->values();

			if ($idsAsignaturasAlumno->isNotEmpty()) {
				$asignaturas = Asignatura::query()
					->whereIn('id', $idsAsignaturasAlumno)
					->orderBy('nombre')
					->get(['id', 'nombre']);
			}
		}

		// Último recurso: todas las asignaturas (para no dejar el selector vacío)
		if ($asignaturas->isEmpty()) {
			$asignaturas = Asignatura::query()->orderBy('nombre')->get(['id', 'nombre']);
		}

		// Selección por defecto: primera asignatura disponible si no viene en la request
		if (!$asignaturaId && $asignaturas->isNotEmpty()) {
			$asignaturaId = (int) $asignaturas->first()->id;
		}

		// Años disponibles para el alumno y, si hay, la asignatura seleccionada
		$aniosQuery = Calificacion::query()->where('alumno_id', $alumno->id);
		if ($asignaturaId) {
			$aniosQuery->where('asignatura_id', $asignaturaId);
		}
		$aniosDisponibles = $aniosQuery->distinct()->orderBy('anio', 'desc')->pluck('anio')->filter()->values();

		// Año seleccionado (param) o, si no viene, tomar el más reciente disponible; en su defecto, año actual
		$anio = $request->integer('anio') ?: null;
		if (!$anio) {
			$anio = $aniosDisponibles->first() ?: (int) date('Y');
		}

		// Construir mapa de notas por mes (1..12) para el alumno y asignatura seleccionada
		// Ahora almacenamos arrays de calificaciones para cada mes
		$notasPorMes = array_fill(1, 12, []);
		$promedio = null;
		$promedioComentario = null;
		$periodoSeleccionado = null;
		$promediosPeriodo = collect();

		if ($asignaturaId) {
			$registros = Calificacion::query()
				->where('alumno_id', $alumno->id)
				->where('asignatura_id', $asignaturaId)
				->where('anio', $anio)
				->orderBy('mes', 'asc')
				->orderBy('numero', 'asc')
				->get(['mes', 'nota', 'numero', 'comentario']);

			foreach ($registros as $reg) {
				$mes = (int) $reg->mes;
				if ($mes >= 1 && $mes <= 12) {
					$notasPorMes[$mes][] = [
						'nota' => $reg->nota,
						'numero' => $reg->numero,
						'comentario' => $reg->comentario,
					];
				}
			}

			// Obtener promedio manual desde calificacion_periodo (periodo actual si existiera, si no el último por fecha_fin)
			$hoy = now()->toDateString();
			$periodoSeleccionado = Periodo::query()
				->whereDate('fecha_inicio', '<=', $hoy)
				->whereDate('fecha_fin', '>=', $hoy)
				->orderBy('fecha_fin', 'desc')
				->first();

			if (!$periodoSeleccionado) {
				$periodoSeleccionado = Periodo::query()->orderBy('fecha_fin', 'desc')->first();
			}

			// Listar todos los promedios disponibles para el alumno y asignatura
			$promediosPeriodo = CalificacionPeriodo::with('periodo')
				->where('alumno_id', $alumno->id)
				->where('asignatura_id', $asignaturaId)
				->get();

			// Si hay un periodo activo o el último, tomarlo para mostrar destacado (opcional)
			if ($periodoSeleccionado) {
				$califPeriodo = $promediosPeriodo->firstWhere('periodo_id', optional($periodoSeleccionado)->id);
			} else {
				$califPeriodo = $promediosPeriodo->sortByDesc(function ($row) {
					return optional($row->periodo)->fecha_fin;
				})->first();
			}

			if ($califPeriodo) {
				$promedio = $califPeriodo->nota;
				$promedioComentario = $califPeriodo->comentario;
			}
		}

		// Fuente para el layout tipo tabla (una sola fila con meses + promedio)
		// Pasamos los arrays de calificaciones para que el layout pueda mostrarlas todas
		$resumen = [[
			'enero' => $notasPorMes[1],
			'febrero' => $notasPorMes[2],
			'marzo' => $notasPorMes[3],
			'abril' => $notasPorMes[4],
			'mayo' => $notasPorMes[5],
			'junio' => $notasPorMes[6],
			'julio' => $notasPorMes[7],
			'agosto' => $notasPorMes[8],
			'septiembre' => $notasPorMes[9],
			'octubre' => $notasPorMes[10],
			'noviembre' => $notasPorMes[11],
			'diciembre' => $notasPorMes[12],
		]];

		return [
			'alumno' => $this->alumno,
			'grupo' => $this->grupo,
			'grupo_id' => $grupoId,
			'asignaturas' => $asignaturas,
			'asignaturas_options' => $asignaturas->pluck('nombre', 'id')->toArray(),
			'asignatura_id' => $asignaturaId,
			'anio' => $anio,
			'anios' => $aniosDisponibles,
			'anios_options' => $aniosDisponibles->mapWithKeys(fn ($a) => [$a => $a])->toArray(),
			'resumen' => collect($resumen),
			'periodo' => $periodoSeleccionado,
			'promedio_periodo' => $promedio,
			'promedio_comentario' => $promedioComentario,
			'promedios_periodo' => $promediosPeriodo->sortByDesc(function ($row) {
				return optional($row->periodo)->fecha_fin;
			})->values(),
		];
	}

	public function name(): ?string
	{
		$nombre = $this->alumno && $this->alumno->user
			? $this->alumno->user->name . ' ' . ($this->alumno->user->apellido ?? '')
			: 'Alumno';
		return 'Ver calificaciones: ' . $nombre;
	}

	public function description(): ?string
	{
		return 'Seleccione la asignatura para ver las calificaciones mensuales y el promedio del año actual.';
	}

	public function permission(): ?iterable
	{
		return null;
	}

	public function commandBar(): iterable
	{
		$actions = [];

		if ($this->grupo) {
			$actions[] = Link::make(__('Volver a Alumnos del Grupo'))
				->icon('bs.arrow-left')
				->route('platform.systems.grupos.alumnos', $this->grupo->id);
		} else {
			$actions[] = Link::make(__('Volver a Alumnos'))
				->icon('bs.arrow-left')
				->route('platform.systems.alumnos');
		}

		return $actions;
	}

	public function layout(): iterable
	{
		return [
			SelectorAsignaturaAnioLayout::class,
			VerCalificacionesMensualesLayout::class,
			PromediosPeriodoLayout::class,
		];
	}

	public function filtrar(Request $request)
	{
		$alumnoId = $request->route('alumno');
		return redirect()->route('platform.systems.alumnos.ver_calificaciones', [
			'alumno' => $alumnoId,
			'grupo_id' => $request->get('grupo_id'),
			'asignatura_id' => $request->get('asignatura_id'),
			'anio' => $request->get('anio'),
		]);
	}
}


