<?php

declare(strict_types=1);

namespace App\Orchid\Screens\CalificacionPeriodo;

use App\Models\CalificacionPeriodo;
use App\Orchid\Layouts\CalificacionPeriodo\CalificacionPeriodoEditLayout;
use Illuminate\Http\Request;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\Screen;
use Orchid\Support\Color;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Toast;

class CalificacionPeriodoEditScreen extends Screen
{
    public $calificacionPeriodo;

    public function query(Request $request): iterable
    {
        $this->calificacionPeriodo = CalificacionPeriodo::where('alumno_id', $request->get('alumno_id'))
            ->where('asignatura_id', $request->get('asignatura_id'))
            ->where('docente_id', $request->get('docente_id'))
            ->where('periodo_id', $request->get('periodo_id'))
            ->first() ?? new CalificacionPeriodo();
        
        $this->calificacionPeriodo->load(['alumno.user', 'asignatura', 'docente.user', 'periodo']);
        return ['calificacionPeriodo' => $this->calificacionPeriodo];
    }

    public function name(): ?string
    {
        return $this->calificacionPeriodo->exists ? 'Editar Calificación por Período' : 'Crear Calificación por Período';
    }

    public function description(): ?string
    {
        return 'Información de la calificación por período académico.';
    }

    public function permission(): ?iterable
    {
        return null;
    }

    public function commandBar(): iterable
    {
        return [
            Button::make(__('Eliminar'))
                ->icon('bs.trash3')
                ->confirm(__('¿Está seguro?'))
                ->method('remove')
                ->canSee($this->calificacionPeriodo->exists),
            Button::make(__('Guardar'))
                ->icon('bs.check-circle')
                ->method('save'),
        ];
    }

    public function layout(): iterable
    {
        return [
            Layout::block(CalificacionPeriodoEditLayout::class)
                ->title(__('Información de la Calificación por Período'))
                ->commands(
                    Button::make(__('Guardar'))
                        ->type(Color::BASIC)
                        ->icon('bs.check-circle')
                        ->canSee($this->calificacionPeriodo->exists)
                        ->method('save')
                ),
        ];
    }

    public function save(Request $request)
    {
        $request->validate([
            'calificacionPeriodo.alumno_id' => 'required|exists:alumnos,id',
            'calificacionPeriodo.asignatura_id' => 'required|exists:asignaturas,id',
            'calificacionPeriodo.docente_id' => 'required|exists:docentes,id',
            'calificacionPeriodo.periodo_id' => 'required|exists:periodos,id',
            'calificacionPeriodo.nota' => 'nullable|numeric|min:0|max:10',
            'calificacionPeriodo.comentario' => 'nullable|string',
        ]);

        CalificacionPeriodo::updateOrCreate(
            [
                'alumno_id' => $request->get('calificacionPeriodo')['alumno_id'],
                'asignatura_id' => $request->get('calificacionPeriodo')['asignatura_id'],
                'docente_id' => $request->get('calificacionPeriodo')['docente_id'],
                'periodo_id' => $request->get('calificacionPeriodo')['periodo_id'],
            ],
            $request->get('calificacionPeriodo')
        );

        Toast::info(__('Calificación por período guardada correctamente'));
        return redirect()->route('platform.systems.calificaciones-periodo');
    }

    public function remove(Request $request)
    {
        CalificacionPeriodo::where('alumno_id', $request->get('alumno_id'))
            ->where('asignatura_id', $request->get('asignatura_id'))
            ->where('docente_id', $request->get('docente_id'))
            ->where('periodo_id', $request->get('periodo_id'))
            ->delete();
        Toast::info(__('Calificación por período eliminada correctamente'));
        return redirect()->route('platform.systems.calificaciones-periodo');
    }
}

