<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Ciclo;

use App\Models\Ciclo;
use App\Orchid\Layouts\Ciclo\CicloEditLayout;
use Illuminate\Http\Request;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\Screen;
use Orchid\Support\Color;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Toast;

class CicloEditScreen extends Screen
{
    public $ciclo;

    public function query(Ciclo $ciclo): iterable
    {
        return ['ciclo' => $ciclo];
    }

    public function name(): ?string
    {
        return $this->ciclo->exists ? 'Editar Ciclo' : 'Crear Ciclo';
    }

    public function description(): ?string
    {
        return 'Información del ciclo.';
    }

    public function permission(): ?iterable
    {
        return null; // Sin restricción de permisos
    }

    public function commandBar(): iterable
    {
        return [
            Button::make(__('Eliminar'))
                ->icon('bs.trash3')
                ->confirm(__('¿Está seguro de eliminar este ciclo?'))
                ->method('remove')
                ->canSee($this->ciclo->exists),
            Button::make(__('Guardar'))
                ->icon('bs.check-circle')
                ->method('save'),
        ];
    }

    public function layout(): iterable
    {
        return [
            Layout::block(CicloEditLayout::class)
                ->title(__('Información del Ciclo'))
                ->description(__('Ingrese la información del ciclo.'))
                ->commands(
                    Button::make(__('Guardar'))
                        ->type(Color::BASIC)
                        ->icon('bs.check-circle')
                        ->canSee($this->ciclo->exists)
                        ->method('save')
                ),
        ];
    }

    public function save(Ciclo $ciclo, Request $request)
    {
        $request->validate([
            'ciclo.nombre' => 'required|string|max:100',
        ]);

        $ciclo->fill($request->get('ciclo'))->save();
        Toast::info(__('Ciclo guardado correctamente'));
        return redirect()->route('platform.systems.ciclos');
    }

    public function remove(Ciclo $ciclo)
    {
        $ciclo->delete();
        Toast::info(__('Ciclo eliminado correctamente'));
        return redirect()->route('platform.systems.ciclos');
    }
}

