<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Colegio;

use App\Models\Colegio;
use App\Orchid\Layouts\Colegio\ColegioEditLayout;
use Illuminate\Http\Request;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\Screen;
use Orchid\Support\Color;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Toast;

class ColegioEditScreen extends Screen
{
    /**
     * @var Colegio
     */
    public $colegio;

    /**
     * Fetch data to be displayed on the screen.
     *
     * @return array
     */
    public function query(Colegio $colegio): iterable
    {
        return [
            'colegio' => $colegio,
        ];
    }

    /**
     * The name of the screen displayed in the header.
     */
    public function name(): ?string
    {
        return $this->colegio->exists ? 'Editar Colegio' : 'Crear Colegio';
    }

    /**
     * Display header description.
     */
    public function description(): ?string
    {
        return 'Información del colegio.';
    }

    public function permission(): ?iterable
    {
        return [
            'platform.systems.colegios',
        ];
    }

    /**
     * The screen's action buttons.
     *
     * @return \Orchid\Screen\Action[]
     */
    public function commandBar(): iterable
    {
        return [
            Button::make(__('Eliminar'))
                ->icon('bs.trash3')
                ->confirm(__('¿Está seguro de eliminar este colegio?'))
                ->method('remove')
                ->canSee($this->colegio->exists),

            Button::make(__('Guardar'))
                ->icon('bs.check-circle')
                ->method('save'),
        ];
    }

    /**
     * The screen's layout elements.
     *
     * @return \Orchid\Screen\Layout[]|string[]
     */
    public function layout(): iterable
    {
        return [
            Layout::block(ColegioEditLayout::class)
                ->title(__('Información del Colegio'))
                ->description(__('Ingrese la información del colegio.'))
                ->commands(
                    Button::make(__('Guardar'))
                        ->type(Color::BASIC)
                        ->icon('bs.check-circle')
                        ->canSee($this->colegio->exists)
                        ->method('save')
                ),
        ];
    }

    /**
     * @param Colegio $colegio
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function save(Colegio $colegio, Request $request)
    {
        $request->validate([
            'colegio.nombre' => 'required|string|max:255',
            'colegio.direccion' => 'nullable|string|max:255',
        ]);

        $colegio->fill($request->get('colegio'))->save();

        Toast::info(__('Colegio guardado correctamente'));

        return redirect()->route('platform.systems.colegios');
    }

    /**
     * @param Colegio $colegio
     * @return \Illuminate\Http\RedirectResponse
     */
    public function remove(Colegio $colegio)
    {
        $colegio->delete();

        Toast::info(__('Colegio eliminado correctamente'));

        return redirect()->route('platform.systems.colegios');
    }
}
