<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Docente;

use App\Models\Docente;
use App\Orchid\Layouts\Docente\DocenteEditLayout;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\Screen;
use Orchid\Support\Color;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Toast;

class DocenteEditScreen extends Screen
{
    public $docente;

    public function query(Docente $docente): iterable
    {
        $docente->load(['user']);
        
        // Cargar asignaturas sin grupo específico (grupo_id es null)
        // Esto representa las asignaturas que el docente puede dictar en general
        $asignaturasSinGrupo = $docente->asignaturas()
            ->wherePivotNull('grupo_id')
            ->get();
        
        // Cargar grupos asignados al docente (grupos únicos)
        $gruposAsignados = $docente->grupos()
            ->select('grupos.id')
            ->distinct()
            ->pluck('grupos.id')
            ->toArray();
        
        return [
            'docente' => $docente,
            'docente.asignaturas' => $asignaturasSinGrupo->pluck('id')->toArray(),
            'docente.grupos' => $gruposAsignados,
        ];
    }

    public function name(): ?string
    {
        return $this->docente->exists ? 'Editar Docente' : 'Crear Docente';
    }

    public function description(): ?string
    {
        return 'Información del docente.';
    }

    public function permission(): ?iterable
    {
        return null; // Sin restricción de permisos
    }

    public function commandBar(): iterable
    {
        return [
            Button::make(__('Eliminar'))
                ->icon('bs.trash3')
                ->confirm(__('¿Está seguro de eliminar este docente?'))
                ->method('remove')
                ->canSee($this->docente->exists),
            Button::make(__('Guardar'))
                ->icon('bs.check-circle')
                ->method('save'),
        ];
    }

    public function layout(): iterable
    {
        return [
            Layout::block(DocenteEditLayout::class)
                ->title(__('Información del Docente'))
                ->description(__('Ingrese la información del docente.'))
                ->commands(
                    Button::make(__('Guardar'))
                        ->type(Color::BASIC)
                        ->icon('bs.check-circle')
                        ->canSee($this->docente->exists)
                        ->method('save')
                ),
        ];
    }

    public function save(Docente $docente, Request $request)
    {
        $request->validate([
            'docente.user_id' => 'required|exists:users,id|unique:docentes,user_id,' . $docente->id,
            'docente.fecha_ingreso' => 'nullable|date',
            'docente.observaciones' => 'nullable|string',
            'docente.asignaturas' => 'nullable|array',
            'docente.asignaturas.*' => 'exists:asignaturas,id',
            'docente.grupos' => 'nullable|array',
            'docente.grupos.*' => 'exists:grupos,id',
        ]);

        $docente->fill($request->get('docente'))->save();

        // Obtener datos del formulario
        $asignaturas = $request->input('docente.asignaturas', []);
        $asignaturas = is_array($asignaturas) ? array_filter($asignaturas) : [];
        
        $gruposSeleccionados = $request->input('docente.grupos', []);
        $gruposSeleccionados = is_array($gruposSeleccionados) ? array_filter($gruposSeleccionados) : [];
        
        // Obtener grupos actualmente asignados (únicos)
        $gruposActuales = $docente->grupos()->distinct()->pluck('grupos.id')->toArray();
        
        // Eliminar relaciones de grupos que ya no están seleccionados
        $gruposAEliminar = array_diff($gruposActuales, $gruposSeleccionados);
        if (!empty($gruposAEliminar)) {
            foreach ($gruposAEliminar as $grupoId) {
                // Eliminar todas las relaciones de este grupo con este docente
                DB::table('docente_asignatura_grupos')
                    ->where('docente_id', $docente->id)
                    ->where('grupo_id', $grupoId)
                    ->delete();
            }
        }
        
        // Sincronizar asignaturas y grupos
        if (!empty($asignaturas)) {
            // Eliminar todas las relaciones existentes del docente (tanto con grupo como sin grupo)
            // para evitar duplicados y luego recrear solo las necesarias
            DB::table('docente_asignatura_grupos')
                ->where('docente_id', $docente->id)
                ->delete();
            
            if (!empty($gruposSeleccionados)) {
                // Si hay grupos seleccionados, crear relaciones asignatura-grupo
                foreach ($gruposSeleccionados as $grupoId) {
                    foreach ($asignaturas as $asignaturaId) {
                        DB::table('docente_asignatura_grupos')->insert([
                            'docente_id' => $docente->id,
                            'asignatura_id' => $asignaturaId,
                            'grupo_id' => $grupoId,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }
                }
            } else {
                // Si no hay grupos seleccionados, crear relaciones con grupo_id = null
                foreach ($asignaturas as $asignaturaId) {
                    DB::table('docente_asignatura_grupos')->insert([
                        'docente_id' => $docente->id,
                        'asignatura_id' => $asignaturaId,
                        'grupo_id' => null,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }
            }
        } else {
            // Si no hay asignaturas seleccionadas, eliminar todas las relaciones
            DB::table('docente_asignatura_grupos')
                ->where('docente_id', $docente->id)
                ->delete();
        }

        Toast::info(__('Docente guardado correctamente'));
        return redirect()->route('platform.systems.docentes');
    }

    public function remove(Docente $docente)
    {
        $docente->delete();
        Toast::info(__('Docente eliminado correctamente'));
        return redirect()->route('platform.systems.docentes');
    }
}

