<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Grupo;

use App\Models\ColegioCiclo;
use App\Models\Grupo;
use App\Orchid\Layouts\Grupo\GrupoEditLayout;
use Illuminate\Http\Request;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\Screen;
use Orchid\Support\Color;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Toast;

class GrupoEditScreen extends Screen
{
    public $grupo;

    public function query(Grupo $grupo): iterable
    {
        $grupo->load(['colegioCiclo.colegio', 'colegioCiclo.ciclo', 'turno']);
        
        // Si el grupo existe, agregar los IDs de colegio y ciclo para los campos
        if ($grupo->exists && $grupo->colegioCiclo) {
            $grupo->colegio_id = $grupo->colegioCiclo->colegio_id;
            $grupo->ciclo_id = $grupo->colegioCiclo->ciclo_id;
        }
        
        return ['grupo' => $grupo];
    }

    public function name(): ?string { return $this->grupo->exists ? 'Editar Grupo' : 'Crear Grupo'; }
    public function description(): ?string { return 'Información del grupo.'; }
    public function permission(): ?iterable { return null; }

    public function commandBar(): iterable
    {
        return [
            Button::make(__('Eliminar'))->icon('bs.trash3')->confirm(__('¿Está seguro?'))->method('remove')->canSee($this->grupo->exists),
            Button::make(__('Guardar'))->icon('bs.check-circle')->method('save'),
        ];
    }

    public function layout(): iterable
    {
        return [
            Layout::block(GrupoEditLayout::class)
                ->title(__('Información del Grupo'))
                ->commands(Button::make(__('Guardar'))->type(Color::BASIC)->icon('bs.check-circle')->canSee($this->grupo->exists)->method('save')),
        ];
    }

    public function save(Grupo $grupo, Request $request)
    {
        $request->validate([
            'grupo.colegio_id' => 'required|exists:colegios,id',
            'grupo.ciclo_id' => 'required|exists:ciclos,id',
            'grupo.turno_id' => 'required|exists:turnos,id',
            'grupo.nombre_grupo' => 'required|string|max:255',
        ]);

        $data = $request->get('grupo');
        
        // Buscar o crear la relación ColegioCiclo
        $colegioCiclo = ColegioCiclo::firstOrCreate(
            [
                'colegio_id' => $data['colegio_id'],
                'ciclo_id' => $data['ciclo_id'],
            ]
        );

        // Asignar el colegio_ciclo_id al grupo
        $grupo->colegio_ciclo_id = $colegioCiclo->id;
        $grupo->turno_id = $data['turno_id'];
        $grupo->nombre_grupo = $data['nombre_grupo'];
        $grupo->save();

        Toast::info(__('Grupo guardado correctamente'));
        return redirect()->route('platform.systems.grupos');
    }

    public function remove(Grupo $grupo)
    {
        $grupo->delete();
        Toast::info(__('Grupo eliminado correctamente'));
        return redirect()->route('platform.systems.grupos');
    }
}

