<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Grupo;

use App\Models\Grupo;
use App\Orchid\Layouts\Grupo\GrupoListLayout;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Orchid\Screen\Actions\Link;
use Orchid\Screen\Screen;
use Orchid\Support\Facades\Toast;

class GrupoListScreen extends Screen
{
    public function query(): iterable
    {
        $user = Auth::user();
        $query = Grupo::with(['colegioCiclo.colegio', 'colegioCiclo.ciclo', 'turno']);
        
        // Si el usuario autenticado es un docente, filtrar solo sus grupos
        if ($user && $user->docente) {
            $docente = $user->docente;
            // Obtener los IDs de los grupos asignados al docente
            $gruposIds = $docente->grupos()
                ->select('grupos.id')
                ->distinct()
                ->pluck('grupos.id')
                ->toArray();
            
            // Filtrar solo los grupos del docente
            if (!empty($gruposIds)) {
                $query->whereIn('grupos.id', $gruposIds);
            } else {
                // Si el docente no tiene grupos asignados, no mostrar ninguno
                $query->whereRaw('1 = 0');
            }
        }
        
        return ['grupos' => $query->defaultSort('id', 'desc')->paginate()];
    }

    public function name(): ?string { return 'Gestión de Grupos'; }
    public function description(): ?string { return 'Lista de grupos registrados.'; }
    public function permission(): ?iterable { return null; }

    public function commandBar(): iterable
    {
        $user = Auth::user();
        $commandBar = [];
        
        // Solo mostrar el botón "Agregar" si el usuario NO es un docente
        if (!$user || !$user->docente) {
            $commandBar[] = Link::make(__('Agregar'))->icon('bs.plus-circle')->route('platform.systems.grupos.create');
        }
        
        return $commandBar;
    }

    public function layout(): iterable { return [GrupoListLayout::class]; }

    public function remove(Request $request): void
    {
        Grupo::findOrFail($request->get('id'))->delete();
        Toast::info(__('Grupo eliminado correctamente'));
    }
}

