<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Padre;

use App\Models\Padre;
use App\Orchid\Layouts\Padre\PadreEditLayout;
use App\Orchid\Layouts\Padre\PadreAlumnosLayout;
use Illuminate\Http\Request;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\Screen;
use Orchid\Support\Color;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Toast;

class PadreEditScreen extends Screen
{
    public $padre;

    public function query(Padre $padre): iterable
    {
        $padre->load('user');
        
        // Cargar alumnos relacionados solo si el padre ya existe
        $alumnos = $padre->exists 
            ? $padre->alumnos()->with('user')->orderBy('id', 'desc')->get()
            : collect();
        
        return [
            'padre' => $padre,
            'alumnos' => $alumnos,
        ];
    }

    public function name(): ?string { return $this->padre->exists ? 'Editar Padre' : 'Crear Padre'; }
    public function description(): ?string { return 'Información del padre.'; }
    public function permission(): ?iterable { return null; }

    public function commandBar(): iterable
    {
        return [
            Button::make(__('Eliminar'))->icon('bs.trash3')->confirm(__('¿Está seguro?'))->method('remove')->canSee($this->padre->exists),
            Button::make(__('Guardar'))->icon('bs.check-circle')->method('save'),
        ];
    }

    public function layout(): iterable
    {
        return [
            Layout::block(PadreEditLayout::class)
                ->title(__('Información del Padre'))
                ->commands(Button::make(__('Guardar'))->type(Color::BASIC)->icon('bs.check-circle')->canSee($this->padre->exists)->method('save')),
            
            Layout::block(PadreAlumnosLayout::class)
                ->title(__('Alumnos Relacionados'))
                ->description(__('Lista de alumnos asignados a este padre. Para editar las relaciones, debe hacerlo desde el menú de cada alumno.'))
                ->canSee($this->padre->exists),
        ];
    }

    public function save(Padre $padre, Request $request)
    {
        $request->validate([
            'padre.user_id' => 'required|exists:users,id|unique:padres,user_id,' . $padre->id,
            'padre.tipo_relacion' => 'nullable|string|max:50',
            'padre.observaciones' => 'nullable|string',
        ]);
        $padre->fill($request->get('padre'))->save();
        Toast::info(__('Padre guardado correctamente'));
        return redirect()->route('platform.systems.padres');
    }

    public function remove(Padre $padre)
    {
        $padre->delete();
        Toast::info(__('Padre eliminado correctamente'));
        return redirect()->route('platform.systems.padres');
    }
}

