<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Periodo;

use App\Models\Periodo;
use App\Orchid\Layouts\Periodo\PeriodoEditLayout;
use Illuminate\Http\Request;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\Screen;
use Orchid\Support\Color;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Toast;

class PeriodoEditScreen extends Screen
{
    public $periodo;

    public function query(Periodo $periodo): iterable
    {
        return ['periodo' => $periodo];
    }

    public function name(): ?string
    {
        return $this->periodo->exists ? 'Editar Período' : 'Crear Período';
    }

    public function description(): ?string
    {
        return 'Información del período académico.';
    }

    public function permission(): ?iterable
    {
        return null;
    }

    public function commandBar(): iterable
    {
        return [
            Button::make(__('Eliminar'))
                ->icon('bs.trash3')
                ->confirm(__('¿Está seguro de eliminar este período?'))
                ->method('remove')
                ->canSee($this->periodo->exists),
            Button::make(__('Guardar'))
                ->icon('bs.check-circle')
                ->method('save'),
        ];
    }

    public function layout(): iterable
    {
        return [
            Layout::block(PeriodoEditLayout::class)
                ->title(__('Información del Período'))
                ->description(__('Ingrese la información del período académico.'))
                ->commands(
                    Button::make(__('Guardar'))
                        ->type(Color::BASIC)
                        ->icon('bs.check-circle')
                        ->canSee($this->periodo->exists)
                        ->method('save')
                ),
        ];
    }

    public function save(Periodo $periodo, Request $request)
    {
        $request->validate([
            'periodo.nombre' => 'required|string|max:100',
            'periodo.fecha_inicio' => 'required|date',
            'periodo.fecha_fin' => 'required|date|after:periodo.fecha_inicio',
        ]);

        $periodo->fill($request->get('periodo'))->save();
        Toast::info(__('Período guardado correctamente'));
        return redirect()->route('platform.systems.periodos');
    }

    public function remove(Periodo $periodo)
    {
        $periodo->delete();
        Toast::info(__('Período eliminado correctamente'));
        return redirect()->route('platform.systems.periodos');
    }
}

