<?php

declare(strict_types=1);

namespace App\Orchid\Screens\Turno;

use App\Models\Turno;
use App\Orchid\Layouts\Turno\TurnoEditLayout;
use Illuminate\Http\Request;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\Screen;
use Orchid\Support\Color;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Toast;

class TurnoEditScreen extends Screen
{
    public $turno;

    public function query(Turno $turno): iterable
    {
        return ['turno' => $turno];
    }

    public function name(): ?string
    {
        return $this->turno->exists ? 'Editar Turno' : 'Crear Turno';
    }

    public function description(): ?string
    {
        return 'Información del turno.';
    }

    public function permission(): ?iterable
    {
        return null; // Sin restricción de permisos
    }

    public function commandBar(): iterable
    {
        return [
            Button::make(__('Eliminar'))
                ->icon('bs.trash3')
                ->confirm(__('¿Está seguro de eliminar este turno?'))
                ->method('remove')
                ->canSee($this->turno->exists),
            Button::make(__('Guardar'))
                ->icon('bs.check-circle')
                ->method('save'),
        ];
    }

    public function layout(): iterable
    {
        return [
            Layout::block(TurnoEditLayout::class)
                ->title(__('Información del Turno'))
                ->description(__('Ingrese la información del turno.'))
                ->commands(
                    Button::make(__('Guardar'))
                        ->type(Color::BASIC)
                        ->icon('bs.check-circle')
                        ->canSee($this->turno->exists)
                        ->method('save')
                ),
        ];
    }

    public function save(Turno $turno, Request $request)
    {
        $request->validate([
            'turno.nombre' => 'required|string|max:100',
        ]);

        $turno->fill($request->get('turno'))->save();
        Toast::info(__('Turno guardado correctamente'));
        return redirect()->route('platform.systems.turnos');
    }

    public function remove(Turno $turno)
    {
        $turno->delete();
        Toast::info(__('Turno eliminado correctamente'));
        return redirect()->route('platform.systems.turnos');
    }
}

