<?php

declare(strict_types=1);

namespace App\Orchid\Screens\User;

use App\Orchid\Layouts\User\UserEditLayout;
use App\Orchid\Layouts\User\UserPasswordLayout;
use App\Orchid\Layouts\User\UserRoleLayout;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Orchid\Access\Impersonation;
use App\Models\User;
use App\Models\Rol;
use Orchid\Screen\Action;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\Screen;
use Orchid\Support\Color;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Toast;

class UserEditScreen extends Screen
{
    /**
     * @var User
     */
    public $user;

    /**
     * Fetch data to be displayed on the screen.
     *
     * @return array
     */
    public function query(User $user): iterable
    {
        $user->load(['roles']);

        // Preparar el nombre del archivo actual para mostrar (solo si el usuario existe)
        $fotoActual = null;
        if ($user->exists) {
            if ($user->foto && Storage::disk('public')->exists($user->foto)) {
                $fotoActual = basename($user->foto);
            } else {
                $fotoActual = 'No hay foto de perfil';
            }
        }

        return [
            'user' => $user,
            'user.foto_actual' => $fotoActual,
        ];
    }

    /**
     * The name of the screen displayed in the header.
     */
    public function name(): ?string
    {
        return $this->user->exists ? 'Edit User' : 'Create User';
    }

    /**
     * Display header description.
     */
    public function description(): ?string
    {
        return 'User profile and privileges. Permissions are inherited from assigned roles.';
    }

    public function permission(): ?iterable
    {
        return [
            'platform.systems.users',
        ];
    }

    /**
     * The screen's action buttons.
     *
     * @return Action[]
     */
    public function commandBar(): iterable
    {
        return [
            Button::make(__('Impersonate user'))
                ->icon('bg.box-arrow-in-right')
                ->confirm(__('You can revert to your original state by logging out.'))
                ->method('loginAs')
                ->canSee($this->user->exists && $this->user->id !== \request()->user()->id),

            Button::make(__('Remove'))
                ->icon('bs.trash3')
                ->confirm(__('Once the account is deleted, all of its resources and data will be permanently deleted. Before deleting your account, please download any data or information that you wish to retain.'))
                ->method('remove')
                ->canSee($this->user->exists),

            Button::make(__('Save'))
                ->icon('bs.check-circle')
                ->method('save'),
        ];
    }

    /**
     * @return \Orchid\Screen\Layout[]
     */
    public function layout(): iterable
    {
        return [

            Layout::block(UserEditLayout::class)
                ->title(__('Profile Information'))
                ->description(__('Update your account\'s profile information and email address.'))
                ->commands(
                    Button::make(__('Save'))
                        ->type(Color::BASIC)
                        ->icon('bs.check-circle')
                        ->canSee($this->user->exists)
                        ->method('save')
                ),

            Layout::block(UserPasswordLayout::class)
                ->title(__('Password'))
                ->description(__('Ensure your account is using a long, random password to stay secure.'))
                ->commands(
                    Button::make(__('Save'))
                        ->type(Color::BASIC)
                        ->icon('bs.check-circle')
                        ->canSee($this->user->exists)
                        ->method('save')
                ),

            Layout::block(UserRoleLayout::class)
                ->title(__('Roles'))
                ->description(__('A Role defines a set of tasks a user assigned the role is allowed to perform. The user will inherit all permissions from the assigned roles. To modify permissions, edit the role in the Roles section.'))
                ->commands(
                    Button::make(__('Save'))
                        ->type(Color::BASIC)
                        ->icon('bs.check-circle')
                        ->canSee($this->user->exists)
                        ->method('save')
                ),

        ];
    }

    /**
     * @return \Illuminate\Http\RedirectResponse
     */
    public function save(User $user, Request $request)
    {
        $request->validate([
            'user.email' => [
                'required',
                Rule::unique(User::class, 'email')->ignore($user),
            ],
            'user.roles' => $user->exists ? 'nullable|array|min:1' : 'required|array|min:1',
            'user.roles.*' => 'exists:roles,id',
            'user.foto' => 'nullable|file|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $user->when($request->filled('user.password'), function (Builder $builder) use ($request) {
            $builder->getModel()->password = Hash::make($request->input('user.password'));
        });

        // Los permisos siempre serán un array vacío, ya que todos los permisos
        // se obtienen de los roles asignados
        $user
            ->fill($request->collect('user')->except(['password', 'permissions', 'roles', 'foto'])->toArray())
            ->forceFill(['permissions' => []])
            ->save();

        // Asignar roles (de aquí vendrán todos los permisos)
        // Si se enviaron roles, actualizarlos; si el usuario existe y no se enviaron, mantener los actuales
        $roles = $request->input('user.roles');
        if (!empty($roles)) {
            $user->replaceRoles($roles);
        } elseif (!$user->exists) {
            // Esto no debería pasar porque la validación requiere roles para usuarios nuevos
            return back()->withErrors(['user.roles' => 'At least one role is required.'])->withInput();
        }

        // Manejar la subida de la foto de perfil (después de guardar el usuario para tener el ID)
        $file = null;
        
        // Intentar obtener el archivo de diferentes maneras posibles
        if ($request->hasFile('user.foto')) {
            $file = $request->file('user.foto');
        } elseif ($request->hasFile('user') && is_array($request->file('user')) && isset($request->file('user')['foto'])) {
            $file = $request->file('user')['foto'];
        } else {
            // Buscar en todos los archivos del request
            $allFiles = $request->allFiles();
            foreach ($allFiles as $key => $uploadedFile) {
                if (is_array($uploadedFile)) {
                    if (isset($uploadedFile['foto'])) {
                        $file = $uploadedFile['foto'];
                        break;
                    }
                    // Buscar recursivamente
                    foreach ($uploadedFile as $subKey => $subFile) {
                        if ($subKey === 'foto' && $subFile && is_object($subFile) && method_exists($subFile, 'isValid')) {
                            $file = $subFile;
                            break 2;
                        }
                    }
                } elseif ($key === 'foto' || (is_object($uploadedFile) && method_exists($uploadedFile, 'isValid'))) {
                    $file = $uploadedFile;
                    break;
                }
            }
        }
        
        if ($file && $file->isValid()) {
            // Obtener el rol del usuario (usar el primer rol o el rol principal)
            $rolesIds = $user->roles()->pluck('id')->toArray();
            
            $rolSlug = 'sin-rol';
            if (!empty($rolesIds)) {
                $rol = Rol::find($rolesIds[0]);
                if ($rol) {
                    $rolSlug = $rol->slug ?? strtolower(str_replace(' ', '-', $rol->name));
                }
            }
            
            // Usar el ID del usuario (ahora ya existe porque se guardó antes)
            $identificador = $user->id;
            
            // Crear la ruta del directorio: upload/fotos-perfil/{rol}/{identificador-usuario}
            $directorio = "upload/fotos-perfil/{$rolSlug}/{$identificador}";
            
            // Asegurar que el directorio existe
            if (!Storage::disk('public')->exists($directorio)) {
                Storage::disk('public')->makeDirectory($directorio, 0755, true);
            }
            
            // Nombre del archivo con timestamp para evitar problemas de caché
            $timestamp = time();
            $nombreArchivo = "avatar-{$timestamp}.jpg";
            
            // Eliminar la foto anterior si existe
            if ($user->foto && Storage::disk('public')->exists($user->foto)) {
                Storage::disk('public')->delete($user->foto);
            }
            
            // Eliminar cualquier archivo en el directorio (por si hay archivos antiguos con otro nombre)
            if (Storage::disk('public')->exists($directorio)) {
                $archivos = Storage::disk('public')->files($directorio);
                foreach ($archivos as $archivo) {
                    Storage::disk('public')->delete($archivo);
                }
            }
            
            // Verificar si GD está disponible para conversión
            $gdAvailable = extension_loaded('gd') && function_exists('imagecreatefromjpeg');
            
            if ($gdAvailable) {
                // Intentar convertir la imagen a JPG usando GD
                $rutaCompleta = Storage::disk('public')->path($directorio . '/' . $nombreArchivo);
                $mimeType = $file->getMimeType();
                $imagenOrigen = null;
                
                try {
                    if ($mimeType === 'image/jpeg' || $mimeType === 'image/jpg') {
                        $imagenOrigen = imagecreatefromjpeg($file->getRealPath());
                    } elseif ($mimeType === 'image/png') {
                        $imagenOrigen = imagecreatefrompng($file->getRealPath());
                    } elseif ($mimeType === 'image/gif') {
                        $imagenOrigen = imagecreatefromgif($file->getRealPath());
                    }
                    
                    if ($imagenOrigen) {
                        // Crear una imagen en blanco con las mismas dimensiones
                        $ancho = imagesx($imagenOrigen);
                        $alto = imagesy($imagenOrigen);
                        $imagenJpg = imagecreatetruecolor($ancho, $alto);
                        
                        // Si es PNG o GIF, usar fondo blanco
                        if ($mimeType === 'image/png' || $mimeType === 'image/gif') {
                            $blanco = imagecolorallocate($imagenJpg, 255, 255, 255);
                            imagefill($imagenJpg, 0, 0, $blanco);
                        }
                        
                        // Copiar la imagen
                        imagecopyresampled($imagenJpg, $imagenOrigen, 0, 0, 0, 0, $ancho, $alto, $ancho, $alto);
                        
                        // Guardar como JPG con calidad 90
                        if (imagejpeg($imagenJpg, $rutaCompleta, 90)) {
                            // Liberar memoria
                            imagedestroy($imagenOrigen);
                            imagedestroy($imagenJpg);
                            
                            // Guardar la ruta relativa en el campo foto
                            $user->foto = $directorio . '/' . $nombreArchivo;
                            $user->save();
                        } else {
                            // Si falla la conversión, guardar el archivo directamente
                            $rutaArchivo = $file->storeAs($directorio, $nombreArchivo, 'public');
                            if ($rutaArchivo && Storage::disk('public')->exists($rutaArchivo)) {
                                $user->foto = $rutaArchivo;
                                $user->save();
                            }
                        }
                    } else {
                        // Si no se puede crear la imagen, guardar directamente
                        $rutaArchivo = $file->storeAs($directorio, $nombreArchivo, 'public');
                        if ($rutaArchivo && Storage::disk('public')->exists($rutaArchivo)) {
                            $user->foto = $rutaArchivo;
                            $user->save();
                        }
                    }
                } catch (\Exception $e) {
                    // Si hay algún error, guardar el archivo directamente
                    $rutaArchivo = $file->storeAs($directorio, $nombreArchivo, 'public');
                    if ($rutaArchivo && Storage::disk('public')->exists($rutaArchivo)) {
                        $user->foto = $rutaArchivo;
                        $user->save();
                    }
                }
            } else {
                // Si GD no está disponible, guardar el archivo directamente
                // Determinar la extensión correcta basada en el tipo MIME
                $mimeType = $file->getMimeType();
                $extension = 'jpg';
                
                if ($mimeType === 'image/png') {
                    $extension = 'png';
                } elseif ($mimeType === 'image/gif') {
                    $extension = 'gif';
                } elseif ($mimeType === 'image/jpeg' || $mimeType === 'image/jpg') {
                    $extension = 'jpg';
                } else {
                    // Si no reconocemos el tipo, usar la extensión original
                    $extension = $file->getClientOriginalExtension() ?: 'jpg';
                }
                
                // Guardar con el nombre avatar-{timestamp}.{extension} para evitar problemas de caché
                $timestamp = time();
                $nombreArchivoFinal = "avatar-{$timestamp}.{$extension}";
                $rutaArchivo = $file->storeAs($directorio, $nombreArchivoFinal, 'public');
                
                if ($rutaArchivo && Storage::disk('public')->exists($rutaArchivo)) {
                    $user->foto = $rutaArchivo;
                    $user->save();
                }
            }
        }

        Toast::info(__('User was saved. All permissions come from assigned roles.'));

        return redirect()->route('platform.systems.users');
    }

    /**
     * @throws \Exception
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function remove(User $user)
    {
        $user->delete();

        Toast::info(__('User was removed'));

        return redirect()->route('platform.systems.users');
    }

    /**
     * @return \Illuminate\Http\RedirectResponse
     */
    public function loginAs(User $user)
    {
        Impersonation::loginAs($user);

        Toast::info(__('You are now impersonating this user'));

        return redirect()->route(config('platform.index'));
    }
}
